using System;
using System.IO;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Azure.WebJobs;
using Microsoft.Azure.WebJobs.Extensions.Http;
using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Logging;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using System.Reflection;
using RestSharp;
using static German_Register_API.Models.FrontEnd;
using System.Collections.Generic;
using German_Register_API.Models;
using ApiHookSkeleton.Models;

namespace ApiHookSkeleton
{
    public static class DDApihookFunction
    {

        private static RestClient RestClient = new RestClient("https://api.openbrewerydb.org/");

        public static string CapitalizeFirstLetter(string s)
        {
            if (String.IsNullOrEmpty(s))
                return s;
            if (s.Length == 1)
                return s.ToUpper();
            return s.Remove(1).ToUpper() + s.Substring(1);
        }


        public static FrontEnd.FrontEndStructure ParseResults(BackEnd.BusinessInfo businessInfo)
        {


            try
            {

                var d = new FrontEndStructure();

               
                d.CompanyInformation.NameOfCompany = businessInfo.name;

                d.CompanyInformation.CompanyAddressStreetNameAndNumber = businessInfo.street;
                d.CompanyInformation.CompanyAddressZipCodeAndCity = businessInfo.postal_code + " " + businessInfo.city;

                d.CompanyInformation.CompanyAddressFull = d.CompanyInformation.CompanyAddressStreetNameAndNumber + ", " + d.CompanyInformation.CompanyAddressZipCodeAndCity;

                d.CompanyInformation.ObjectOfTheCompany = CapitalizeFirstLetter(businessInfo.brewery_type);
                d.CompanyInformation.Country = businessInfo.country;
                d.CompanyInformation.Phone = businessInfo.phone;
                d.CompanyInformation.State = businessInfo.state;

                return d;
            }
            catch (Exception ex)
            {

                return null;
            }


        }




        public static DD_Api_Structure GenerateSampleData()
        {
            var d = new DD_Api_Structure();
            d.Structure =  new FrontEndStructure();
            d.Structure.CompanyInformation = new Example_CompanyInformation();

            d.Suggestions = new List<DD_Suggestion_Model>();
            d.Suggestions.Add(new DD_Suggestion_Model() { FieldReference = "CompanyInformation.NameOfCompany", FieldSample = "The Brewer's Art / Old Line Brewery LLC" });

            d.FilterValues = new List<DD_FilterValues_Model>();


            var dict = new Dictionary<string, DDPortalApiDataModel>
            {
                {"CompanyInformation.NameOfCompany", new DDPortalApiDataModel() { Description="Returns the name of the company.", DisplayName="Name of company", Usage= UsageTypes.OnlyDataFields } },
                {"CompanyInformation.CompanyAddressStreetNameAndNumber", new DDPortalApiDataModel(){ Description= "Returns the street name and number for the companys registered address in the format Examplestreet 99.", DisplayName="Companys address (street name and number)", Usage=UsageTypes.OnlyDataFields} },
                {"CompanyInformation.CompanyAddressZipCodeAndCity", new DDPortalApiDataModel() { Description = "Returns the zip code and city name for the companys registered address in the format 9999 City.", DisplayName="Companys address (zip code and city)", Usage= UsageTypes.OnlyDataFields} },
                {"CompanyInformation.CompanyAddressFull",  new DDPortalApiDataModel() { Description="Returns the full company address in the format c/o John Doe, Examplestreet 99, 9999 City.", DisplayName="Companys full address (c/o name (if available),street name, no., zip code and city)", Usage=UsageTypes.OnlyDataFields}},
                {"CompanyInformation.ObjectOfTheCompany", new DDPortalApiDataModel(){ Description="Returns the type of brewery", DisplayName = "Type of brewery", Usage=UsageTypes.OnlyDataFields} },
                {"CompanyInformation.Country", new DDPortalApiDataModel(){ Description="Returns the companys country", DisplayName = "Country", Usage=UsageTypes.OnlyDataFields} },
                {"CompanyInformation.Phone", new DDPortalApiDataModel(){ Description="Returns the companys phone number", DisplayName = "Phonenumber", Usage=UsageTypes.OnlyDataFields} },
                {"CompanyInformation.State", new DDPortalApiDataModel(){ Description="Returns the companys state", DisplayName = "State", Usage=UsageTypes.OnlyDataFields} },


            };

            d.Dictionary = dict;

            return d;


        }

        [SimpleBearerAuthorize("abc1234")]
        [FunctionName("Search")]
        public static async Task<IActionResult> Run(
            [HttpTrigger(AuthorizationLevel.Anonymous, "get", "post", Route = null)] HttpRequest req,
            ILogger log)
        {
            string name = req.Query["search"];

            if (string.IsNullOrEmpty(name) )
            {
                return new BadRequestResult();
            }

            if (name=="draftersample")
            {
                return new OkObjectResult(JsonConvert.SerializeObject(GenerateSampleData()));
            }

            var request = new RestRequest("breweries/search?query=" + name);
            try
            {
                var result = await RestClient.ExecuteAsync<List<Models.BackEnd.SearchModel>>(request);
                if (result.IsSuccessful)
                {
                    return new OkObjectResult(JsonConvert.SerializeObject(result.Data));
                }
                else
                    return new NotFoundResult();
            }
            catch (Exception e)
            {

                return new BadRequestResult();
            }
            
        }

        [SimpleBearerAuthorize("abc1234")]
        [FunctionName("GetBrewery")]
        public static async Task<IActionResult> GetBrewery(
            [HttpTrigger(AuthorizationLevel.Anonymous, "get", "post", Route = null)] HttpRequest req,
            ILogger log)
        {
            string id = req.Query["search"];

            if (string.IsNullOrEmpty(id))
            {
                return new BadRequestResult();
            }

            var request = new RestRequest("v1/breweries/" + id);
            try
            {
                var result = await RestClient.ExecuteAsync<Models.BackEnd.BusinessInfo>(request);
                if (result.IsSuccessful)
                {
                    var outputlist = new List<FrontEnd.FrontEndStructure>();

                    outputlist.Add(ParseResults(result.Data));

                    return new OkObjectResult(JsonConvert.SerializeObject(outputlist));
                }
                else
                    return new NotFoundResult();
            }
            catch (Exception e)
            {

                return new BadRequestResult();
            }

        }
    }
}
